Attribute VB_Name = "modPnMHex"
' TK3 programming function for new PIC types introduced by PIC n' Mix
' decoupled from the rest of TK3 to ease maintenance
' Written by Andrew Jarvis 15.08.04
'
' Revision History
' 26.03.04 Initial Revision
' 15.08.04 Final Revision
'

Option Explicit

'representation of PIC memory address
Type ADDRESS_TYPE
    'the location content
    data As Byte
    'flag set if content changed
    dirty As Boolean
End Type

'hex file record constants
'ignore other record types
Private Const HEX_DATA_RECORD = 0
Private Const HEX_END_RECORD = 1

'number of ASCII characters that comprise hex data
Private Const BYTE_SIZE = 2
Private Const WORD_SIZE = 4

'byte order constants
Public Const WORD_LSB = 0
Public Const WORD_MSB = 1

'start of the data bytes within a hex record
Private Const DATA_START = 10

'program memory buffer
Public ProgMem() As ADDRESS_TYPE
'EE memory buffer
Public EEPROMMem() As ADDRESS_TYPE
'config memory buffer
Public ConfigMem() As ADDRESS_TYPE
'user id memory
Public IDMem() As ADDRESS_TYPE

'pic sizes doubled up for hex locations
Private hexPicSize As Integer
Private hexDataLimit As Integer

Public Sub PnM_LoadHexFile()

    'size the memory structures
    InitialiseMemory
    'load the file into memory
    ReadHexFile
    
End Sub
Public Function PnM_LastAddress(ByVal align As Byte, Optional ByVal ignoreCalibration As Boolean = False) As Integer
    
    'find the last used address, to optimise programming loop - otherwise we
    'have to load the whole address range
    Dim x As Integer
    Dim address As Integer
    
    'hexpicsize is the overall size of the pic, so the last instruction to be
    'written is actually picsize - 2. However, some PICs store calibration data
    'at this location which we might want to ignore when looking for the last
    'user programmed location
    address = hexPicSize - 2
    If ignoreCalibration Then address = address - 2
    
    'search back through the memory map
    For x = address To 0 Step -2
        If ProgMem(x).dirty And ProgMem(x + 1).dirty Then
            ' not a blank position...
            Exit For
        End If
    Next x
    
    'x holds the last location, but the number of bytes programmed will
    'actually be x+2
    x = x + 2
    
    'some programming algorithms require alignement on an n-byte boundary,
    'where n is typically 1, 4, 8.
    align = align * 2

    'align to word boundary
    If x Mod align <> 0 Then
        x = x + (align - (x Mod align))
    End If

    '...and return the answer
    PnM_LastAddress = x - 2
    
End Function
Public Function PnM_HexSize() As Integer

    PnM_HexSize = hexPicSize
    
End Function
Private Sub InitialiseMemory()
  
    Dim x As Integer
    hexPicSize = PICsize * 2
    hexDataLimit = &H4200 + (eeprom% * 2)
    
    'size the program memory buffer
    ReDim ProgMem(hexPicSize - 1)
    'then intialise it to empty
    For x = 0 To hexPicSize - 1 Step 2
        ProgMem(x).data = &HFF
        ProgMem(x + 1).data = &H3F
    Next x
    
    'size the EE memory buffer
    'no need to initialise the eeprom, since we only overwrite
    'it if there are embedded directives in the hex file
    ReDim EEPROMMem(eeprom% - 1)
    
    'size the configuration memory buffer.
    ReDim ConfigMem(1)
    'initialise to empty
    ConfigMem(WORD_MSB).data = &H3F
    ConfigMem(WORD_LSB).data = &HFF
    
    'size the UserID memory buffer.
    ReDim IDMem(4)

End Sub
Private Sub ReadHexFile()

    'various forward declarations of variables
    Dim x As Integer
    'active hex file
    Dim hexFile As Integer
    'the hex file record, 1 per line
    Dim hexRecord As String
    'number of data bytes in a record
    Dim numBytes As Integer
    'the address as specified in the hex file
    Dim hexAddress As Integer
    'record type, only processing data and end records
    Dim recordType As Integer
    'incremental address as bytes are parsed
    Dim dataAddress As Integer
    'working data byte from hex record
    Dim dataByte As Byte
     
    'open the active hex file
    hexFile = FreeFile
    Open FileName For Input As hexFile
 
    'load the hex file into memory.
    'the mpasm assembler help gives a good description of hex file format
    'also see PnM September/October 04
    'each record starts with :BBAAAATT, where
    'BB = 2 digits representing the byte count for the line
    'AAAA = 4 digit hex address representing the start address of the record
    'TT = 2 digit record type
    
    ' read the hex file one record at a time until the end of the file is reached,
    ' or the record type signals end of file.
    Do Until (EOF(hexFile)) Or (recordType = HEX_END_RECORD)
    
        Line Input #hexFile, hexRecord
        
        'look for the record mark
        If Left$(hexRecord, 1) = ":" Then
            'decode the remainder of the prefix, starting with the count
            numBytes = Val("&H" & Mid$(hexRecord, 2, BYTE_SIZE))
            'start address
            hexAddress = Val("&H" & Mid$(hexRecord, 4, WORD_SIZE))
            'record type
            recordType = Val("&H" & Mid$(hexRecord, 8, BYTE_SIZE))
              
            'interrogate recordType. if it's a data record then we have work to do.
            'if it's an end of file record then we're finished.
            'if it's a segment or linear address record we're also finished...
            If recordType = HEX_DATA_RECORD Then
            
                For x = 0 To numBytes - 1
                    'determine the current working address
                    dataAddress = hexAddress + x
                    'get the data byte by figuring the hex value from the file
                    dataByte = Val("&H" & Mid$(hexRecord, DATA_START + BYTE_SIZE * x, BYTE_SIZE))
                    
                    'populate memory structures based on the decoded address
                    'note: values in comments below are the PIC addresses, not hex.
                    
                    '1. Program Memory (0 <= address < hexPICSize)
                    If (dataAddress < hexPicSize) Then
                        PnM_SetAddress ProgMem(dataAddress), dataByte
                        
                    '2. ID Locations (2000 <= address < 2004)
                    ElseIf (dataAddress >= &H4000 And dataAddress < &H4008) Then
                        PnM_SetAddress IDMem(dataAddress - &H4000), dataByte
                     
                    ' locations 2004, 2005 are reserved - and 2006 is the device ID
                    
                    '3. Configuration Word (2007 = address)
                    ElseIf (dataAddress >= &H400E And dataAddress < &H4010) Then
                        PnM_SetAddress ConfigMem(dataAddress - &H400E), dataByte
                        
                    '4. EEPROM Memory (2100 <= address < hexDataLimit)
                    'All the other memory addresses hold words (hi and lo bytes)
                    'EEPROM is 8 bit which means only one data byte per address location,
                    'so only read bytes from even address locations (exactly divisible by 2)
                    ElseIf (dataAddress >= &H4200 And dataAddress < hexDataLimit And dataAddress Mod 2 = 0) Then
                        PnM_SetAddress EEPROMMem((dataAddress - &H4200) \ 2), dataByte
                    End If
                Next x
                
            End If
        End If
    Loop
    
    Close #hexFile
    
End Sub
Public Sub PnM_SetAddress(ByRef addr As ADDRESS_TYPE, ByVal data As Byte)
'sets the data byte at a given address location and marks it as dirty
'dirty flag is used to optimise programming speed later.
    addr.data = data
    addr.dirty = True
End Sub
